/*
 * @(#)ZipArchive.java	1.0 98/12/11
 *
 * Java Project performed by
 *     Frank Moisant
 *    Patrick Titiano
 *        (ITRA3)
 *
 */


import java.io.IOException;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.File;

import java.util.Vector;
import java.util.Enumeration;
import java.util.Date;
import java.util.Hashtable;

import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipOutputStream;
import java.util.zip.ZipInputStream;

/**
 * This class is used to manage Zip archive, it can create, add files, delete files
 * in the archive
 * @version 1.0
 * @author Frank Moisant - Patrick Titiano
 */
public class ZipArchive {
    
    // private members
    private String zipFileName;
    private Vector table;
    private boolean zipFileOpen;


    /**
     * ZipArchive constructor, it initialize the zipArchive
     */
    public ZipArchive() {
	zipFileOpen = false;
    }
    
    
    /**
     * This method select an archive an call the setTable() method
     * @param zipFileName the path + name of the archive to open
     * @return true if the archive exist and not comrupted
     */
    public boolean setArchiveName(String zipFileName) {
	this.zipFileName = zipFileName;
	table = new Vector();
	zipFileOpen = setTable();
	return zipFileOpen;
    }


    /**
     * This method return the state of zipArchive
     * @return true if an archive is opened
     */
    public boolean isSelected() {
	return zipFileOpen;
    }


    /**
     * This method return the table to display, with all file information
     * @return the table to dispay
     */
    public Vector getTable() {
	return table;
    }


    /**
     * This method extract the information of the archive, and put them in a vector
     * of rows. ready to display.
     * @return true if the archive is valid
     */
    private boolean setTable() {

	// create zip streams and read all entries to put in entries vector
	ZipInputStream zipFile;
	FileInputStream inFile;
	Enumeration enum;
	Vector entries = new Vector();
	ZipEntry tmpEntry;
	try {
	    inFile = new FileInputStream(zipFileName);
	    zipFile = new ZipInputStream(inFile);
	    while((tmpEntry = zipFile.getNextEntry()) != null) {
		entries.addElement(tmpEntry);
	    }
	}
	catch(IOException e) {
	    return false;
	}
	try {
	    zipFile.close();
	}
	catch(IOException e) {
	    System.out.println("Can't close the file");
	}
	if(entries.size() == 0) 
	    return false;

	// enumerate all entries and get all information
	enum = entries.elements();
	while(enum.hasMoreElements()) {
	    ZipEntry zipEntry = (ZipEntry )enum.nextElement();
	    if(!zipEntry.isDirectory()) {
		Vector row = new Vector();

		// extract the filename and pathname
		String fullName = zipEntry.getName();
		int lastSeparatorIndex = fullName.lastIndexOf("/");
		String fileName;
		String pathName;
		if(lastSeparatorIndex == -1){
		    fileName = fullName;
		    pathName = "";
		}
		else {
		    fileName = fullName.substring(lastSeparatorIndex+1);
		    pathName = fullName.substring(0,lastSeparatorIndex);
		}
		int lastPointIndex = fileName.lastIndexOf(".");
		
		// extract the file extention to display good icon
		String extension;
		if(lastPointIndex == -1) {
		    extension = "UNKNOWN";
		}
		else {
		    extension = fileName.substring(lastPointIndex+1).toUpperCase();
		}
		row.addElement(fileName);
		row.addElement(extension);

		// extract the date
		String theDate = (new Date(zipEntry.getTime())).toString();
		row.addElement(theDate.substring(0,theDate.length()));

		// extract the real size, compressed size, and calcul the ratio
		long realSize = zipEntry.getSize();
		long compressedSize = zipEntry.getCompressedSize();
		row.addElement(Long.toString(realSize));
		long ratio = 0;
		if(realSize != 0)
		    ratio = -100*(compressedSize-realSize)/realSize;
		row.addElement(Long.toString(ratio) + " %");
		row.addElement(Long.toString(compressedSize));
		row.addElement(pathName);

		// add the row at the table
		table.addElement(row);
	    }
	}
	return true;
    }

    /**
     * This method delete all selected files in the archive
     * @param toDelete vector of name of entries
     * @return true if the operation finish successfully
     */
    public boolean deleteFilesFromArchive(Vector toDelete) {

	// create zip archive stream
	FileOutputStream fileOutputStream = null;
	ZipOutputStream zipOutputStream = null;
	FileInputStream fileInputStream = null;
	ZipInputStream zipInputStream = null;

	// rename the current archive to copy it
	File tmpfile = new File("tmp.zip");
	new File(zipFileName).renameTo(tmpfile);

	// open zip archive
	try {
	    fileOutputStream = new FileOutputStream(zipFileName);
	    zipOutputStream = new ZipOutputStream(fileOutputStream);
	}
	catch(IOException e) {
	    zipFileOpen = false;
	    return false;
	}

	// save the data of tmp.zip in the new archive without selected files
	try {
	    fileInputStream = new FileInputStream("tmp.zip");
	    zipInputStream = new ZipInputStream(fileInputStream);

	    // for all entries read the entry
	    ZipEntry tmpEntry;
	    while((tmpEntry = zipInputStream.getNextEntry()) != null) {

		// the entry name is not in toDelete slection
		if(!toDelete.contains(tmpEntry.getName())) {

		    // copy the content of the entry in the new archive
		    byte buffer[] = new byte[512];
		    zipOutputStream.putNextEntry(tmpEntry);
		    int size;
		    while((size = zipInputStream.read(buffer)) != -1) {
			zipOutputStream.write(buffer,0,size);
		    }
		    zipOutputStream.closeEntry();
		}
	    }
	    zipInputStream.close();
	}
	catch(IOException e) {
	    System.out.println("Close Error");
	}
	try {
	    zipOutputStream.close();
	}
	catch(IOException e) {
	    System.out.println("zip");
	    zipFileOpen = false;
	    return false;
	}
	
	// delete tmp zip and call setArchiveName to refresh the table
	new File("tmp.zip").delete();
	return setArchiveName(zipFileName);
    }



    /**
     * Create a new archive with selected files
     * @param name the name of the archive to create
     * @param filesToAdd vector of file names to add in this new archive
     * @param positionInPathName index of the path should have been truncated, to save relative path
     * @return true if the operation finish successfully
     */
    public boolean addFileToTheNewArchive(String name, Vector filesToAdd, int positionInPathName) { 

	// create zip streams
	FileOutputStream fileOutputStream = null;
	ZipOutputStream zipOutputStream = null;
	try {
	    fileOutputStream = new FileOutputStream(name);
	    zipOutputStream = new ZipOutputStream(fileOutputStream);
	}
	catch(IOException e) {
	    zipFileOpen = false;
	    return false;
	}

	// for all elements to add
	Enumeration enum = filesToAdd.elements();
	while(enum.hasMoreElements()) {

	    // get the file name
	    String currentfile = (String)enum.nextElement();
	    FileInputStream file;
	    try {
		byte buffer[] = new byte[512];

		// create an stream from the file to add
		file = new FileInputStream(currentfile);

		// extract the pathname to send to the zipEntry
		if(currentfile.charAt(positionInPathName) == File.separator.charAt(0))
		    positionInPathName++;
		ZipEntry zipEntry = new ZipEntry(currentfile.replace(File.separator.charAt(0),'/').substring(positionInPathName));

		// put the entry
		zipOutputStream.putNextEntry(zipEntry);

		// copy all datas in the file
		int size;
		while((size = file.read(buffer)) != -1) {
		    zipOutputStream.write(buffer,0,size);
		}
		zipOutputStream.closeEntry();
	    }
	    catch(IOException e) {
		System.out.println("File " + currentfile + " not found");
	    }
	}

	try {
	    zipOutputStream.close();
	}
	catch(IOException e) {
	    System.out.println("zip");
	    zipFileOpen = false;
	    return false;
	}

	// refresh the table
	return setArchiveName(name);
    }


    /**
     * Add files into the current zip Archive
     * @param filesToAdd vector of file name
     * @param positionInPathName index of the path should have been truncated, to save relative path
     * @return true if the operation finish successfully
     */
    public boolean addFileToTheArchive(Vector filesToAdd, int positionInPathName) { 

	// create zip streams
	FileOutputStream fileOutputStream = null;
	ZipOutputStream zipOutputStream = null;
	FileInputStream fileInputStream = null;
	ZipInputStream zipInputStream = null;

	// rename the current archive
	File tmpfile = new File("tmp.zip");
	new File(zipFileName).renameTo(tmpfile);
	try {
	    fileOutputStream = new FileOutputStream(zipFileName);
	    zipOutputStream = new ZipOutputStream(fileOutputStream);
	}
	catch(IOException e) {
	    zipFileOpen = false;
	    return false;
	}

	// save the data of tmp.zip in the new archive
	try {
	    fileInputStream = new FileInputStream("tmp.zip");
	    zipInputStream = new ZipInputStream(fileInputStream);
	    
	    // read all entries and copy the content in the new archive
	    ZipEntry tmpEntry;
	    while((tmpEntry = zipInputStream.getNextEntry()) != null) {
		byte buffer[] = new byte[512];
		zipOutputStream.putNextEntry(tmpEntry);
		int size;
		while((size = zipInputStream.read(buffer)) != -1) {
		    zipOutputStream.write(buffer,0,size);
		}
		zipOutputStream.closeEntry();
	    }
	    zipInputStream.close();
	}
	catch(IOException e) {
	    System.out.println("Close Error");
	}

	// for all elements in the filesToAdd
	Enumeration enum = filesToAdd.elements();
	while(enum.hasMoreElements()) {

	    // get the current file name
	    String currentfile = (String)enum.nextElement();
	    FileInputStream file;
	    try {
		byte buffer[] = new byte[512];

		// open the file to copy it
		file = new FileInputStream(currentfile);

		// extract name of the entry
		if(currentfile.charAt(positionInPathName) == File.separator.charAt(0))
		    positionInPathName++;

		// create the entry and put it in the new archive
		ZipEntry zipEntry = new ZipEntry(currentfile.replace(File.separator.charAt(0),'/').substring(positionInPathName));
		zipOutputStream.putNextEntry(zipEntry);

		// copy the datas
		int size;
		while((size = file.read(buffer)) != -1) {
		    zipOutputStream.write(buffer,0,size);
		}
		zipOutputStream.closeEntry();
	    }
	    catch(IOException e) {
		System.out.println("File " + currentfile + " not found");
	    }
	}
	
	try {
	    zipOutputStream.close();
	}
	catch(IOException e) {
	    System.out.println("zip");
	    zipFileOpen = false;
	    return false;
	}
	// delete the tmp file and refresh the table
	new File("tmp.zip").delete();
	return setArchiveName(zipFileName);
    }


    /**
     * Extract file from the current archive
     * @param toExtract vector of entry name to extract
     * @param pathToExtract the directory where extract the selected entries
     * @return true if the operation finish successfully
     */
    public boolean extractFileFromTheArchive(Vector toExtract, String pathToExtract) {

	// declare zip streams
	FileInputStream fileIn;
	ZipInputStream zipIn;

	// create a hashtable to put entry name and path to real file name to extract it
	Hashtable hashtable = new Hashtable();

	// for all elements in toExtract vector
	ZipEntry zipEntry;
	Enumeration enum = toExtract.elements();
	while(enum.hasMoreElements()) {

	    // extract the file name and pathname and put it in the hashtable
	    String theFile = (String)enum.nextElement();
	    String fullName = theFile;
	    int lastSeparatorIndex = fullName.lastIndexOf("/");
	    String fileName;
	    String pathName;
	    if(lastSeparatorIndex == -1){
		fileName = fullName;
		pathName = "";
	    }
	    else {
		fileName = fullName.substring(lastSeparatorIndex+1);
		pathName = fullName.substring(0,lastSeparatorIndex) + File.separator;
	    }
	    lastSeparatorIndex = pathToExtract.lastIndexOf(File.separator);
	    int firstSeparatorIndex = pathToExtract.indexOf(File.separator);
	    String separator = File.separator;
	    if((lastSeparatorIndex == firstSeparatorIndex)&&
	       (firstSeparatorIndex == pathToExtract.length() -1)) {
		separator = "";
	    }

	    // create the necessaries directories
	    String theRealPathNameOfFile =  pathToExtract + separator + pathName.replace('/',File.separator.charAt(0));
	    File creator = new File(theRealPathNameOfFile);
	    creator.mkdirs();
	    hashtable.put(theFile,theRealPathNameOfFile + fileName);
	}
	try {
	    
	    // create zip streams
	    fileIn = new FileInputStream(zipFileName);
	    zipIn = new ZipInputStream(fileIn);

	    // for all entries
	    while((zipEntry = zipIn.getNextEntry()) != null) {
		String theNewFile;
		if((theNewFile = (String) hashtable.get(zipEntry.getName())) != null) {

		    // verify if the file exist, never overwrite it
		    File verify = new File(theNewFile);
		    if(!verify.exists()) {
			FileOutputStream fileOut = null;
			try {

			    // create the file and copy the datas in it
			    fileOut = new FileOutputStream(theNewFile);
			    byte[] buffer = new byte[512];
			    int size;
			    while((size = zipIn.read(buffer)) != -1) {
				fileOut.write(buffer,0,size);
			    }
			    fileOut.close();
			}
			catch(IOException e) {
			    System.out.println("error extract the file");
			}
		    }
		    else {
			System.out.println(theNewFile + " exist");
		    }
		}
	    }
	}
	catch(IOException e) {
	    return false;
	}
	return true;
    }
}
